# -*- coding: utf-8 -*-
"""
    ResolveUrl Kodi Addon
    Copyright (C) 2013 Bstrdsmkr
    Additional fixes by mortael, jairoxyz

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

    Adapted for use in xbmc from:
    https://github.com/beautify-web/js-beautify/blob/master/python/jsbeautifier/unpackers/packer.py

    Unpacker for Dean Edward's p.a.c.k.e.r, a part of javascript beautifier
    by Einar Lielmanis <einar@jsbeautifier.org>

        written by Stefano Sanfilippo <a.little.coder@gmail.com>

    usage:

    if detect(some_string):
        unpacked = unpack(some_string)

    Unpacker for Dean Edward's p.a.c.k.e.r
"""

import re
import binascii
from six import PY2


def detect(source):
    """Detects whether `source` is P.A.C.K.E.R. coded."""
    mystr = re.search(
        r"eval[ ]*\([ ]*function[ ]*\([ ]*p[ ]*,[ ]*a[ ]*,[ ]*c["
        r" ]*,[ ]*k[ ]*,[ ]*e[ ]*,[ ]*",
        source,
    )
    return mystr is not None


def unpack(source):
    """Unpacks P.A.C.K.E.R. packed js code."""
    payload, symtab, radix, count = _filterargs(source)

    if count != len(symtab):
        raise UnpackingError('Malformed p.a.c.k.e.r. symtab.')

    try:
        unbase = Unbaser(radix)
    except TypeError:
        raise UnpackingError('Unknown p.a.c.k.e.r. encoding.')

    def lookup(match):
        """Look up symbols in the synthetic symtab."""
        word = match.group(0)
        return symtab[int(word)] if radix == 1 else symtab[unbase(word)] or word

    def getstring(c, a=radix):
        foo = chr(c % a + 161)
        if c < a:
            return foo
        else:
            return getstring(int(c / a), a) + foo

    payload = payload.replace("\\\\", "\\").replace("\\'", "'")
    p = re.search(r'eval\(function\(p,a,c,k,e.+?String\.fromCharCode\(([^)]+)', source)
    if p:
        pnew = re.findall(r'String\.fromCharCode\(([^)]+)', source)[0].split('+')[1] == '161'
    else:
        pnew = False

    if pnew:
        for i in range(count - 1, -1, -1):
            payload = payload.replace(getstring(i).decode('latin-1') if PY2 else getstring(i), symtab[i])
        return _replacejsstrings((_replacestrings(payload)))
    else:
        source = re.sub(r"\b\w+\b", lookup, payload) if PY2 else re.sub(r"\b\w+\b", lookup, payload, flags=re.ASCII)
        return _replacestrings(source)


def _filterargs(source):
    """Juice from a source file the four args needed by decoder."""
    argsregex = r"}\s*\('(.*)',\s*(.*?),\s*(\d+),\s*'(.*?)'\.split\('\|'\)"
    args = re.search(argsregex, source, re.DOTALL).groups()

    try:
        payload, radix, count, symtab = args
        radix = 36 if not radix.isdigit() else int(radix)
        return payload, symtab.split('|'), radix, int(count)
    except ValueError:
        raise UnpackingError('Corrupted p.a.c.k.e.r. data.')


def _replacestrings(source):
    """Strip string lookup table (list) and replace values in source."""
    match = re.search(r'var *(_\w+)=\["(.*?)"];', source, re.DOTALL)

    if match:
        varname, strings = match.groups()
        startpoint = len(match.group(0))
        lookup = strings.split('","')
        variable = '%s[%%d]' % varname
        for index, value in enumerate(lookup):
            if '\\x' in value:
                value = value.replace('\\x', '')
                value = binascii.unhexlify(value).decode('ascii')
            source = source.replace(variable % index, '"%s"' % value)
        return source[startpoint:]
    return source


def _replacejsstrings(source):
    """Strip JS string encodings and replace values in source."""
    match = re.findall(r'\\x([0-7][0-9A-F])', source)

    if match:
        match = set(match)
        for value in match:
            source = source.replace('\\x{0}'.format(value), binascii.unhexlify(value).decode('ascii'))

    return source


class Unbaser(object):
    """Functor for a given base. Will efficiently convert
    strings to natural numbers."""
    ALPHABET = {
        62: '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ',
        95: (r' !"#$%&\'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ'
             r'[\]^_`abcdefghijklmnopqrstuvwxyz{|}~')
    }

    def __init__(self, base):
        self.base = base

        # If base can be handled by int() builtin, let it do it for us
        if 2 <= base <= 36:
            self.unbase = lambda string: int(string, base)
        else:
            if base < 62:
                self.ALPHABET[base] = self.ALPHABET[62][0:base]
            elif 62 < base < 95:
                self.ALPHABET[base] = self.ALPHABET[95][0:base]
            # Build conversion dictionary cache
            try:
                self.dictionary = dict(
                    (cipher, index) for index, cipher in enumerate(
                        self.ALPHABET[base]))
            except KeyError:
                raise TypeError('Unsupported base encoding.')

            self.unbase = self._dictunbaser

    def __call__(self, string):
        return self.unbase(string)

    def _dictunbaser(self, string):
        """Decodes a  value to an integer."""
        ret = 0
        for index, cipher in enumerate(string[::-1]):
            ret += (self.base ** index) * self.dictionary[cipher]
        return ret


class UnpackingError(Exception):
    """Badly packed source or general error. Argument is a
    meaningful description."""
    pass


if __name__ == "__main__":
    # test = '''eval(function(p,a,c,k,e,d){while(c--)if(k[c])p=p.replace(new RegExp('\\b'+c.toString(a)+'\\b','g'),k[c]);return p}('4(\'30\').2z({2y:\'5://a.8.7/i/z/y/w.2x\',2w:{b:\'2v\',19:\'<p><u><2 d="20" c="#17">2u 19.</2></u><16/><u><2 d="18" c="#15">2t 2s 2r 2q.</2></u></p>\',2p:\'<p><u><2 d="20" c="#17">2o 2n b.</2></u><16/><u><2 d="18" c="#15">2m 2l 2k 2j.</2></u></p>\',},2i:\'2h\',2g:[{14:"11",b:"5://a.8.7/2f/13.12"},{14:"2e",b:"5://a.8.7/2d/13.12"},],2c:"11",2b:[{10:\'2a\',29:\'5://v.8.7/t-m/m.28\'},{10:\'27\'}],26:{\'25-3\':{\'24\':{\'23\':22,\'21\':\'5://a.8.7/i/z/y/\',\'1z\':\'w\',\'1y\':\'1x\'}}},s:\'5://v.8.7/t-m/s/1w.1v\',1u:"1t",1s:"1r",1q:\'1p\',1o:"1n",1m:"1l",1k:\'5\',1j:\'o\',});l e;l k=0;l 6=0;4().1i(9(x){f(6>0)k+=x.r-6;6=x.r;f(q!=0&&k>=q){6=-1;4().1h();4().1g(o);$(\'#1f\').j();$(\'h.g\').j()}});4().1e(9(x){6=-1});4().1d(9(x){n(x)});4().1c(9(){$(\'h.g\').j()});9 n(x){$(\'h.g\').1b();f(e)1a;e=1;}',36,109,'||font||jwplayer|http|p0102895|me|vidto|function|edge3|file|color|size|vvplay|if|video_ad|div||show|tt102895|var|player|doPlay|false||21600|position|skin|test||static|1y7okrqkv4ji||00020|01|type|360p|mp4|video|label|FFFFFF|br|FF0000||deleted|return|hide|onComplete|onPlay|onSeek|play_limit_box|setFullscreen|stop|onTime|dock|provider|391|height|650|width|over|controlbar|5110|duration|uniform|stretching|zip|stormtrooper|213|frequency|prefix||path|true|enabled|preview|timeslidertooltipplugin|plugins|html5|swf|src|flash|modes|hd_default|3bjhohfxpiqwws4phvqtsnolxocychumk274dsnkblz6sfgq6uz6zt77gxia|240p|3bjhohfxpiqwws4phvqtsnolxocychumk274dsnkba36sfgq6uzy3tv2oidq|hd|original|ratio|broken|is|link|Your|such|No|nofile|more|any|availabe|Not|File|OK|previw|jpg|image|setup|flvplayer'.split('|')))'''
    # test = '''eval(function(p,a,c,k,e,d){e=function(c){return(c<a?'':e(parseInt(c/a)))+((c=c%a)>35?String.fromCharCode(c+29):c.toString(36))};if(!''.replace(/^/,String)){while(c--){d[e(c)]=k[c]||e(c)}k=[function(e){return d[e]}];e=function(){return'\\w+'};c=1};while(c--){if(k[c]){p=p.replace(new RegExp('\\b'+e(c)+'\\b','g'),k[c])}}return p}('y.x(A(\'%0%f%b%9%1%d%8%8%o%e%B%c%0%e%d%0%f%w%1%7%3%2%p%d%1%n%2%1%c%0%t%0%f%7%8%8%d%5%6%1%7%e%b%l%7%1%2%e%9%q%c%0%6%1%z%2%0%f%b%1%9%c%0%s%6%6%l%G%4%4%5%5%5%k%b%7%5%8%o%i%2%k%6%i%4%2%3%p%2%n%4%5%7%6%9%s%4%j%q%a%h%a%3%a%E%a%3%D%H%9%K%C%I%m%r%g%h%L%v%g%u%F%r%g%3%J%3%j%3%m%h%4\'));',48,48,'22|72|65|6d|2f|77|74|61|6c|63|4e|73|3d|6f|6e|20|4d|32|76|59|2e|70|51|64|69|62|79|31|68|30|7a|34|66|write|document|75|unescape|67|4f|5a|57|55|3a|44|47|4a|78|49'.split('|'),0,{}))'''
    # test = '''eval(function(p,a,c,k,e,d){e=function(c){return(c<a?'':e(parseInt(c/a)))+((c=c%a)>35?String.fromCharCode(c+29):c.toString(36))};if(!''.replace(/^/,String)){while(c--){d[e(c)]=k[c]||e(c)}k=[function(e){return d[e]}];e=function(){return'\\w+'};c=1};while(c--){if(k[c]){p=p.replace(new RegExp('\\b'+e(c)+'\\b','g'),k[c])}}return p}('x.w(z(\'%1%f%9%b%0%d%7%7%m%e%A%c%1%e%d%1%f%v%0%3%i%2%o%d%0%s%2%0%c%1%q%1%f%3%7%7%d%6%5%0%3%e%9%l%3%0%2%e%b%g%c%1%5%0%y%2%1%f%9%0%b%c%1%r%5%5%l%E%4%4%6%6%6%n%9%3%6%7%m%k%2%n%5%k%4%2%i%o%2%s%4%6%3%5%b%r%4%8%D%h%C%a%F%8%H%B%I%h%i%a%g%8%u%a%q%j%t%j%g%8%t%h%p%j%p%a%G%4\'));',45,45,'72|22|65|61|2f|74|77|6c|5a|73|55|63|3d|6f|6e|20|79|59|6d|4d|76|70|69|2e|62|7a|30|68|64|44|54|66|write|document|75|unescape|67|51|32|6a|3a|35|5f|47|34'.split('|'),0,{}))'''
    test = '''eval(function(p,a,c,k,e,d){e=function(c){return(c<a?'':e(parseInt(c/a)))+((c=c%a)>35?String.fromCharCode(c+29):c.toString(36))};if(!''.replace(/^/,String)){while(c--){d[e(c)]=k[c]||e(c)}k=[function(e){return d[e]}];e=function(){return'\\w+'};c=1};while(c--){if(k[c]){p=p.replace(new RegExp('\\b'+e(c)+'\\b','g'),k[c])}}return p}('q.r(s(\'%h%t%a%p%u%6%c%n%0%5%l%4%2%4%7%j%0%8%1%o%b%3%7%m%1%8%a%7%b%3%d%6%1%f%0%v%1%5%D%9%0%5%c%g%0%4%A%9%0%f%k%z%2%8%1%C%2%i%d%6%2%3%k%j%2%3%y%e%x%w%g%B%E%F%i%h%e\'));',42,42,'5a|4d|4f|54|6a|44|33|6b|57|7a|56|4e|68|55|3e|47|69|65|6d|32|45|46|31|6f|30|75|document|write|unescape|6e|62|6c|2f|3c|22|79|63|66|78|59|72|61'.split('|'),0,{}))'''
    # test = '''eval(function(p,a,c,k,e,d){e=function(c){return(c<a?\'\':e(c/a))+String.fromCharCode(c%a+161)};while(c--){if(k[c]){p=p.replace(new RegExp(e(c),\'g\'),k[c])}}return p}(\'\xd7 \xa1=["\\\\\xa9\\\\\xa5\\\\\xa3\\\\\xa2\\\\\xaa\\\\\xa3\\\\\xaf\\\\\xa9\\\\\xa5\\\\\xb8\\\\\xc5\\\\\xbe\\\\\xbe\\\\\xb9\\\\\xb9\\\\\xc6\\\\\xc0\\\\\xc3\\\\\xb7\\\\\xab\\\\\xc3\\\\\xb9\\\\\xab\\\\\xc0\\\\\xc4\\\\\xc4\\\\\xc4\\\\\xc3\\\\\xa2\\\\\xb7\\\\\xc0\\\\\xab\\\\\xc6\\\\\xab\\\\\xa2\\\\\xc4\\\\\xc0\\\\\xb7\\\\\xc0\\\\\xa5\\\\\xab\\\\\xc6\\\\\xbd","\\\\\xa7\\\\\xa2\\\\\xa3\\\\\xa7\\\\\xb8\\\\\xc5\\\\\xbe\\\\\xbe\\\\\xb9\\\\\xb9\\\\\xc6\\\\\xc0\\\\\xc3\\\\\xb7\\\\\xab\\\\\xc3\\\\\xb9\\\\\xab\\\\\xc0\\\\\xc4\\\\\xc4\\\\\xc4\\\\\xc3\\\\\xa2\\\\\xb7\\\\\xc0\\\\\xab\\\\\xc6\\\\\xab\\\\\xa2\\\\\xc4\\\\\xc0\\\\\xb7\\\\\xc0\\\\\xa5\\\\\xab\\\\\xc6\\\\\xbd","\\\\\xb5\\\\\xa2\\\\\xa3\\\\\xa2\xa7\\\\\xa3\\\\\xa2\\\\\xad","\\\\\xae\\\\\xac\\\\\xa8\\\\\xa2\\\\\xba\\\\\xa6\\\\\xac\\\\\xa2\\\\\xa8","\\\\\xa6\\\\\xaa\\\\\xca\\\\\xae\\\\\xaf\\\\\xaf\\\\\xa4\\\\\xa7\\\\\xa3\\\\\xa2\\\\\xa8","\\\\\xa2\xab\\\\\xb8\\\\\xb1\\\\\xa7\\\\\xa6\\\\\xa8\\\\\xdf\\\\\xa4\\\\\xa5\\\\\xa8\\\\\xa2\\\\\xa7","\\\\\xab\\\\\xa4\\\\\xa7\\\\\xa2","\\\\\xab\\\\\xb2\\\\\xa6","\\\\\xb2\\\\\xa3\\\\\xa3\\\\\xaf\\\\\xaa\\\\\xfd\\\\\xb4\\\\\xb4\\\\\xad\\\\\xb8\\\\\xba\\\\\xa2\\\\\xad\\\\\xa6\\\\\xac\\\\\xa6\\\\\xaa\\\\\xa3\\\\\xc5\\\\\xab\\\\\xa4\\\\\xad\\\\\xb4\\\\\xaf\\\\\xa4\\\\\xaa\\\\\xa3\\\\\xa2\\\\\xa7\\\\\xb4\\\\\xa2\xac\\\\\xae\\\\\xa7\\\\\xa9\\\\\xc9\\\\\xe2\\\\\xf5\\\\\xf4\\\\\xa2\xc8\\\\\xca\\\\\xad\\\\\xb3\\\\\xc3\\\\\xd6\\\\\xf6\\\\\xa8\\\\\xcf\\\\\xab\\\\\xb7\\\\\xea\\\\\xac\\\\\xca\\\\\xb3\\\\\xd0\\\\\xd3\\\\\xa2\xae\\\\\xbd\\\\\xa2\xc9\\\\\xb7\\\\\xd6\\\\\xf5\\\\\xe5\\\\\xac\\\\\xca\\\\\xc8\\\\\xd6\\\\\xff\\\\\xc8\\\\\xb7\\\\\xe8\\\\\xe8\\\\\xc8\\\\\xf5\\\\\xa2\xa7\\\\\xb8\\\\\xc8\\\\\xb3\\\\\xf4\\\\\xcf\\\\\xc8\\\\\xce\\\\\xa3\\\\\xa6\\\\\xca\\\\\xb7\\\\\xe8\\\\\xb7\\\\\xab\\\\\xad\\\\\xd0\\\\\xc8\\\\\xde\\\\\xb8\\\\\xaa\\\\\xb6\\\\\xe1\\\\\xad\\\\\xb2\\\\\xbf\\\\\xea\\\\\xad\\\\\xd0\\\\\xa5\\\\\xd6\\\\\xe2\\\\\xf7\\\\\xa2\xae\\\\\xc8\\\\\xce\\\\\xa3\\\\\xf7\\\\\xe1\\\\\xca\\\\\xc0\\\\\xb6\\\\\xa2\xa1\\\\\xdd\\\\\xd0\\\\\xa6\\\\\xea\\\\\xe6\\\\\xa9\\\\\xaa\\\\\xb1\\\\\xc8\\\\\xd6\\\\\xdf\\\\\xca\\\\\xa2\xab\\\\\xd6\\\\\xe6\\\\\xb1\\\\\xcd\\\\\xa3\\\\\xb8\\\\\xd6\\\\\xad\\\\\xb2\\\\\xe2\\\\\xa2\xa1\\\\\xce\\\\\xc4\\\\\xaa\\\\\xa8\\\\\xa2\xde\\\\\xaf\\\\\xea\\\\\xb1\\\\\xa2\xab\\\\\xf7\\\\\xaf\\\\\xe2\\\\\xe2\\\\\xa8\\\\\xe8\\\\\xa2\xa1\\\\\xa2\xab\\\\\xb2\\\\\xdd\\\\\xa5\\\\\xb3\\\\\xa9\\\\\xe6\\\\\xe2\\\\\xdd\\\\\xea\\\\\xa3\\\\\xa2\\\\\xde\\\\\xe8\\\\\xed\\\\\xea\\\\\xd0\\\\\xe1\\\\\xb8\\\\\xe2\\\\\xe2\\\\\xf6\\\\\xbf\\\\\xca\\\\\xde\\\\\xd6\\\\\xc4\\\\\xca\\\\\xc8\\\\\xa9\\\\\xdd\\\\\xa2\xae\\\\\xa9\\\\\xaf\\\\\xa2\xa7\\\\\xa5\\\\\xbd\\\\\xaf\\\\\xf4\\\\\xa8\\\\\xce\\\\\xa8\\\\\xe2\\\\\xc8\\\\\xac\\\\\xe1\\\\\xc3\\\\\xe1\\\\\xb3\\\\\xab\\\\\xd5\\\\\xff\\\\\xae\\\\\xe5\\\\\xca\\\\\xe8\\\\\xe6\\\\\xf4\\\\\xf9\\\\\xbd\\\\\xd0\\\\\xe8\\\\\xe8\\\\\xb6\\\\\xaa\\\\\xb5\\\\\xa5\\\\\xae\\\\\xf5\\\\\xf4\\\\\xc4\\\\\xcf\\\\\xb9\\\\\xa9\\\\\xb1\\\\\xab\\\\\xf9\\\\\xb7\\\\\xdd","\\\\\xf5\\\\\xae\\\\\xac\\\\\xdd\\\\\xa2\\\\\xb6\\\\\xa2\\\\\xa9\\\\\xa4\\\\\xaf\\\\\xa2\\\\\xa7\\\\\xc5\\\\\xab\\\\\xa4\\\\\xad","\\\\\xb2\\\\\xa3\\\\\xa3\\\\\xaf\\\\\xaa\\\\\xfd\\\\\xb4\\\\\xb4\\\\\xb5\\\\\xae\\\\\xac\\\\\xa8\\\\\xa2\\\\\xb6\\\\\xa2\\\\\xa9\\\\\xa4\\\\\xaf\\\\\xa2\\\\\xa7\\\\\xc5\\\\\xab\\\\\xa4\\\\\xad","\\\\\xff\\\\\xa7\\\\\xa6\\\\\xb5\\\\\xa6\\\\\xac\\\\\xa5\\\\\xa9","\\\\\xb2\\\\\xa9\\\\\xaa","\\\\\xb2\\\\\xa3\\\\\xa3\\\\\xaf\\\\\xaa\\\\\xfd\\\\\xb4\\\\\xb4\\\\\xb9\\\\\xd5\\\\\xb6\\\\\xa4\\\\\xa8\\\\\xbc\\\\\xa5\\\\\xa8\\\\\xa5\\\\\xaf\\\\\xa3\\\\\xa6\\\\\xb6\\\\\xa2\\\\\xc5\\\\\xa5\\\\\xce\\\\\xa5\\\\\xad\\\\\xa5\\\\\xa6\\\\\xe6\\\\\xa2\\\\\xa8\\\\\xc5\\\\\xac\\\\\xa2\\\\\xa3\\\\\xb4\\\\\xa2\\\\\xe5\\\\\xaf\\\\\xc9\\\\\xc3\\\\\xd5\\\\\xb7\\\\\xbe\\\\\xc6\\\\\xc4\\\\\xbe\\\\\xbe\\\\\xbe\\\\\xb9\\\\\xa3\xa2\\\\\xa5\\\\\xab\\\\\xa9\\\\\xc9\\\\\xf9\\\\\xbd\\\\\xd0\\\\\xb3\\\\\xa5\\\\\xb3\\\\\xbd\\\\\xb9\\\\\xb3\\\\\xbd\\\\\xbe\\\\\xbc\\\\\xb9\\\\\xb9\\\\\xa8\\\\\xc0\\\\\xbc\\\\\xbe\\\\\xb9\\\\\xab\\\\\xbd\\\\\xbc\\\\\xb1\\\\\xa8\\\\\xc4\\\\\xc3\\\\\xbc\\\\\xb3\\\\\xb3\\\\\xb9\\\\\xb3\\\\\xbe\\\\\xb1\\\\\xa5\\\\\xa5\\\\\xb1\\\\\xc0\\\\\xb1\\\\\xb1\\\\\xf9\\\\\xbd\\\\\xd0\\\\\xf9\\\\\xbd\\\\\xf2\\\\\xa3\xa2\\\\\xb2\\\\\xad\\\\\xa5\\\\\xab\\\\\xc9\\\\\xa2\\\\\xc3\\\\\xc3\\\\\xbe\\\\\xa2\\\\\xbd\\\\\xb7\\\\\xa8\\\\\xbe\\\\\xb1\\\\\xbe\\\\\xb7\\\\\xb9\\\\\xb3\\\\\xab\\\\\xbd\\\\\xb3\\\\\xc0\\\\\xa8\\\\\xc6\\\\\xc3\\\\\xa5\\\\\xc3\\\\\xb3\\\\\xc4\\\\\xb7\\\\\xab\\\\\xa8\\\\\xb3\\\\\xa2\\\\\xb1\\\\\xd5\\\\\xb9\\\\\xbd\\\\\xb7\\\\\xc6\\\\\xc6\\\\\xc3\\\\\xc3\\\\\xab\\\\\xb3\\\\\xb1\\\\\xc0\\\\\xc0\\\\\xb7\\\\\xa2\\\\\xd5\\\\\xc4\\\\\xb3\\\\\xc0\\\\\xbd\\\\\xb7\\\\\xb7\\\\\xb3\\\\\xbd\\\\\xc6\\\\\xa5\\\\\xc4\\\\\xab\\\\\xbe\\\\\xc6\\\\\xa8\\\\\xbd\\\\\xa8\\\\\xb4\\\\\xb3\\\\\xa5\\\\\xb3\\\\\xbd\\\\\xb9\\\\\xb3\\\\\xbd\\\\\xbe\\\\\xbc\\\\\xb9\\\\\xb9\\\\\xa8\\\\\xc0\\\\\xbc\\\\\xbe\\\\\xb9\\\\\xab\\\\\xbd\\\\\xbc\\\\\xb1\\\\\xa8\\\\\xc4\\\\\xc3\\\\\xbc\\\\\xb3\\\\\xb3\\\\\xb9\\\\\xb3\\\\\xbe\\\\\xb1\\\\\xa5\\\\\xa5\\\\\xb1\\\\\xc0\\\\\xb1\\\\\xb1\\\\\xb4\\\\\xaa\\\\\xa2\\\\\xaf\\\\\xb4\\\\\xb6\\\\\xa6\\\\\xa8\\\\\xa2\\\\\xa4\\\\\xb4\\\\\xa2\\\\\xa2\\\\\xb9\\\\\xab\\\\\xbd\\\\\xb9\\\\\xd5\\\\\xc3\\\\\xec\\\\\xd5\\\\\xbd\\\\\xa8\\\\\xd5\\\\\xb7\\\\\xb9\\\\\xa2\\\\\xb9\\\\\xec\\\\\xa2\\\\\xbe\\\\\xd5\\\\\xb9\\\\\xa2\\\\\xab\\\\\xa5\\\\\xbd\\\\\xec\\\\\xa2\\\\\xa5\\\\\xb9\\\\\xbe\\\\\xba\\\\\xba\\\\\xc3\\\\\xc3\\\\\xb4\\\\\xad\\\\\xa5\\\\\xaa\\\\\xa3\\\\\xa2\\\\\xa7\\\\\xc5\\\\\xad\\\\\xb7\\\\\xae\\\\\xc6","\\\\\xb2\\\\\xa3\\\\\xad\\\\\xa9\\\\\xc4","\\\\\xad\\\\\xa2\\\\\xa3\\\\\xa5\\\\\xa8\\\\\xa5\\\\\xa3\\\\\xa5","\\\\\xae\\\\\xac\\\\\xa6\\\\\xba\\\\\xa4\\\\\xa7\\\\\xad","\\\\\xcb\\\\\xba\\\\\xba\\\\\xba\\\\\xba\\\\\xb3\\\\\xb3","\\\\\xc3\\\\\xd5\\\\\xfd\\\\\xc0","\\\\\xae\\\\\xbf\\\\\xdf\\\\\xcd\\\\\xa8\\\\\xa2\xa7\\\\\xc4\\\\\xed\\\\\xbe\\\\\xab\\\\\xac\\\\\xb3\\\\\xc8\\\\\xa2\xae\\\\\xdf\\\\\xb9\\\\\xc8\\\\\xa2\xa1\\\\\xa9\\\\\xa2\xc8\\\\\xff\\\\\xe1\\\\\xb7\\\\\xe1\\\\\xd3\\\\\xb8\\\\\xac\\\\\xe6\\\\\xe6\\\\\xca\\\\\xe1\\\\\xab\\\\\xe2\\\\\xd6\\\\\xf7\\\\\xb9\\\\\xc6\\\\\xa6\\\\\xc8\\\\\xed\\\\\xf2\\\\\xbf\\\\\xaa\\\\\xb2\\\\\xde\\\\\xdf\\\\\xb7\\\\\xd0\\\\\xa9\\\\\xae\\\\\xaf\\\\\xd3\\\\\xa4\\\\\xbf\\\\\xa8\\\\\xdd\\\\\xbf\\\\\xf4\\\\\xc4\\\\\xae\\\\\xa6\\\\\xbd\\\\\xc0\\\\\xe6","","\\\\\xa9\\\\\xa2\\\\\xac\\\\\xb5\\\\\xa3\\\\\xb2","\\\\\xc5\\\\\xb6\\\\\xa3\\\\\xa3","\\\\\xa6\\\\\xac\\\\\xa8\\\\\xa2\\\\\xe5\\\\\xff\\\\\xba","\\\\\xba\\\\\xa6\\\\\xa9\\\\\xa2","\\\\\xb1\\\\\xbc\\\\\xab\\\\\xa8\\\\\xac","\\\\\xb4\\\\\xb4\\\\\xad\\\\\xb8\\\\\xba\\\\\xa2\\\\\xad\\\\\xa6\\\\\xac\\\\\xa6\\\\\xaa\\\\\xa3\\\\\xc5\\\\\xab\\\\\xa4\\\\\xad\\\\\xb4","\\\\\xb4\\\\\xb4\\\\\xad\\\\\xb8\\\\\xba\\\\\xa2\\\\\xad\\\\\xa6\\\\\xac\\\\\xa6\\\\\xaa\\\\\xa3\\\\\xc5\\\\\xab\\\\\xa4\\\\\xad\\\\\xb4\\\\\xaa\\\\\xae\\\\\xb1\\\\\xa3\\\\\xa6\\\\\xa3\\\\\xa9\\\\\xa2\\\\\xc5\\\\\xb6\\\\\xa3\\\\\xa3\\\\\xa2\xac\\\\\xae\\\\\xa7\\\\\xa9\\\\\xc9","\\\\\xa8\\\\\xa2\\\\\xba\\\\\xa5\\\\\xae\\\\\xa9\\\\\xa3","\\\\\xa9\\\\\xa5\\\\\xb1\\\\\xa2\\\\\xa9","\\\\\xaf\\\\\xae\\\\\xaa\\\\\xb2","\\\\\xa2\\\\\xa5\\\\\xab\\\\\xb2","\\\\\xa3\\\\\xa7\\\\\xa5\\\\\xab\\\\\xce\\\\\xaa","\\\\\xaa\\\\\xa4\\\\\xae\\\\\xa7\\\\\xab\\\\\xa2\\\\\xaa","\\\\\xa3\\\\\xb8\\\\\xaf\\\\\xa2","\\\\\xa9\\\\\xa4\\\\\xb5\\\\\xa4","\\\\\xa3\\\\\xa4\\\\\xaf\\\\\xbc\\\\\xa7\\\\\xa6\\\\\xb5\\\\\xb2\\\\\xa3","\\\\\xaa\\\\\xb2\\\\\xa5\\\\\xa7\\\\\xa6\\\\\xac\\\\\xb5","\\\\\xb4\\\\\xb4\\\\\xad\\\\\xb8\\\\\xba\\\\\xa2\\\\\xad\\\\\xa6\\\\\xac\\\\\xa6\\\\\xaa\\\\\xa3\\\\\xc5\\\\\xab\\\\\xa4\\\\\xad\\\\\xb4\\\\\xa2\\\\\xad\\\\\xb1\\\\\xa2\\\\\xa8\\\\\xb4\\\\\xbd\\\\\xc0\\\\\xba\\\\\xa2\\\\\xb7\\\\\xb3\\\\\xab\\\\\xd5\\\\\xbc\\\\\xb3\\\\\xab\\\\\xbd\\\\\xa8\\\\\xbc\\\\\xbe\\\\\xbd\\\\\xb1\\\\\xa5\\\\\xbc\\\\\xc6\\\\\xbe\\\\\xb3\\\\\xbe\\\\\xbc\\\\\xc0\\\\\xa5\\\\\xc4\\\\\xc6\\\\\xa2\\\\\xb9\\\\\xc4\\\\\xa2\\\\\xb1\\\\\xb9\\\\\xb1\\\\\xba","\\\\\xcc\\\\\xa9\\\\\xa3\\\\\xd2\\\\\xa6\\\\\xba\\\\\xa7\\\\\xa5\\\\\xad\\\\\xa2\\\\\xb0\\\\\xaa\\\\\xa7\\\\\xab\\\\\xc9\\\\\xcc\\\\\xcf\\\\\xae\\\\\xa4\\\\\xa3\\\\\xd2\\\\\xb4\\\\\xb4\\\\\xad\\\\\xb8\\\\\xba\\\\\xa2\\\\\xad\\\\\xa6\\\\\xac\\\\\xa6\\\\\xaa\\\\\xa3\\\\\xc5\\\\\xab\\\\\xa4\\\\\xad\\\\\xb4\\\\\xa2\\\\\xad\\\\\xb1\\\\\xa2\\\\\xa8\\\\\xb4\\\\\xbd\\\\\xc0\\\\\xba\\\\\xa2\\\\\xb7\\\\\xb3\\\\\xab\\\\\xd5\\\\\xbc\\\\\xb3\\\\\xab\\\\\xbd\\\\\xa8\\\\\xbc\\\\\xbe\\\\\xbd\\\\\xb1\\\\\xa5\\\\\xbc\\\\\xc6\\\\\xbe\\\\\xb3\\\\\xbe\\\\\xbc\\\\\xc0\\\\\xa5\\\\\xc4\\\\\xc6\\\\\xa2\\\\\xb9\\\\\xc4\\\\\xa2\\\\\xb1\\\\\xb9\\\\\xb1\\\\\xba\\\\\xcc\\\\\xcf\\\\\xae\\\\\xa4\\\\\xa3\\\\\xd2\\\\\xb0\\\\\xba\\\\\xa7\\\\\xa5\\\\\xad\\\\\xa2\\\\\xb1\\\\\xa4\\\\\xa7\\\\\xa8\\\\\xa2\\\\\xa7\\\\\xc9\\\\\xcc\\\\\xcf\\\\\xae\\\\\xa4\\\\\xa3\\\\\xd2\\\\\xb3\\\\\xcc\\\\\xcf\\\\\xae\\\\\xa4\\\\\xa3\\\\\xd2\\\\\xb0\\\\\xa5\\\\\xa9\\\\\xa9\\\\\xa4\\\\\xbf\\\\\xd0\\\\\xae\\\\\xa9\\\\\xa9\\\\\xca\\\\\xab\\\\\xa7\\\\\xa2\\\\\xa2\\\\\xac\\\\\xc9\\\\\xcc\\\\\xcf\\\\\xae\\\\\xa4\\\\\xa3\\\\\xd2\\\\\xa3\\\\\xa7\\\\\xae\\\\\xa2\\\\\xcc\\\\\xcf\\\\\xae\\\\\xa4\\\\\xa3\\\\\xd2\\\\\xb0\\\\\xbf\\\\\xa2\\\\\xb1\\\\\xce\\\\\xa6\\\\\xa3\\\\\xa5\\\\\xa9\\\\\xa9\\\\\xa4\\\\\xbf\\\\\xba\\\\\xae\\\\\xa9\\\\\xa9\\\\\xaa\\\\\xab\\\\\xa7\\\\\xa2\\\\\xa2\\\\\xac\\\\\xc9\\\\\xcc\\\\\xcf\\\\\xae\\\\\xa4\\\\\xa3\\\\\xd2\\\\\xa3\\\\\xa7\\\\\xae\\\\\xa2\\\\\xcc\\\\\xcf\\\\\xae\\\\\xa4\\\\\xa3\\\\\xd2\\\\\xb0\\\\\xad\\\\\xa4\\\\\xe6\\\\\xa5\\\\\xa9\\\\\xa9\\\\\xa4\\\\\xbf\\\\\xba\\\\\xae\\\\\xa9\\\\\xa9\\\\\xaa\\\\\xab\\\\\xa7\\\\\xa2\\\\\xa2\\\\\xac\\\\\xc9\\\\\xcc\\\\\xcf\\\\\xae\\\\\xa4\\\\\xa3\\\\\xd2\\\\\xa3\\\\\xa7\\\\\xae\\\\\xa2\\\\\xcc\\\\\xcf\\\\\xae\\\\\xa4\\\\\xa3\\\\\xd2\\\\\xb0\\\\\xbf\\\\\xa6\\\\\xa8\\\\\xa3\\\\\xb2\\\\\xc9\\\\\xcc\\\\\xcf\\\\\xae\\\\\xa4\\\\\xa3\\\\\xd2\\\\\xd5\\\\\xbe\\\\\xb3\\\\\xcc\\\\\xcf\\\\\xae\\\\\xa4\\\\\xa3\\\\\xd2\\\\\xb0\\\\\xb2\\\\\xa2\\\\\xa6\\\\\xb5\\\\\xb2\\\\\xa3\\\\\xc9\\\\\xcc\\\\\xcf\\\\\xae\\\\\xa4\\\\\xa3\\\\\xd2\\\\\xb7\\\\\xbd\\\\\xb3\\\\\xcc\\\\\xcf\\\\\xae\\\\\xa4\\\\\xa3\\\\\xd2\\\\\xcc\\\\\xb5\\\\\xa3\\\\\xd2\\\\\xcc\\\\\xa9\\\\\xa3\\\\\xd2\\\\\xb4\\\\\xa6\\\\\xba\\\\\xa7\\\\\xa5\\\\\xad\\\\\xa2\\\\\xcc\\\\\xb5\\\\\xa3\\\\\xd2","\\\\\xca\\\\\xb2\\\\\xa5\\\\\xa7\\\\\xa2\\\\\xb0\\\\\xab\\\\\xb2\\\\\xa6","\\\\\xba\\\\\xa5\\\\\xab\\\\\xa2\\\\\xb1\\\\\xa4\\\\\xa4\\\\\xce","\\\\\xa3\\\\\xbf\\\\\xa6\\\\\xa3\\\\\xa3\\\\\xa2\\\\\xa7","\\\\\xa2\\\\\xad\\\\\xa5\\\\\xa6\\\\\xa9","\\\\\xab\\\\\xa9\\\\\xa2\\\\\xa5\\\\\xa7","\\\\\xa9\\\\\xa4\\\\\xab\\\\\xa5\\\\\xa3\\\\\xa6\\\\\xa4\\\\\xac","\\\\\xa3\\\\\xa4\\\\\xaf","\\\\\xcb","\\\\\xb2\\\\\xa3\\\\\xa3\\\\\xaf\\\\\xaa\\\\\xfd\\\\\xb4\\\\\xb4\\\\\xb5\\\\\xa4\\\\\xa4\\\\\xb5\\\\\xa9\\\\\xa2\\\\\xc5\\\\\xab\\\\\xa4\\\\\xad\\\\\xb4","\\\\\xa5\\\\\xa8\\\\\xa8\\\\\xdf\\\\\xa6\\\\\xaa\\\\\xa3\\\\\xa2\\\\\xac\\\\\xa2\\\\\xa7","\\\\\xa9\\\\\xa5\\\\\xae\\\\\xac\\\\\xab\\\\\xb2","\\\\\xea\\\\\xa5\\\\\xea","\\\\\xaa\\\\\xa2\\\\\xa3\\\\\xa2\xa7\\\\\xa3\\\\\xa2\\\\\xad","\\\\\xab\\\\\xb2\\\\\xa7\\\\\xa4\\\\\xad\\\\\xa2","\\\\\xaa\\\\\xb2\\\\\xa4\\\\\xbf","\\\\\xcb\\\\\xad\\\\\xcd\\\\\xa4\\\\\xac\\\\\xa3\\\\\xa5\\\\\xa6\\\\\xac\\\\\xa2\\\\\xa7\\\\\xec\\\\\xb0\\\\\xcb\\\\\xa9\\\\\xa4\\\\\xa5\\\\\xa8\\\\\xa6\\\\\xac\\\\\xb5","\\\\\xb2\\\\\xa6\\\\\xa8\\\\\xa2","\\\\\xcb\\\\\xb6\\\\\xa6\\\\\xa8\\\\\xa2\\\\\xa4\\\\\xcd\\\\\xa4\\\\\xac\\\\\xa3\\\\\xa5\\\\\xa6\\\\\xac\\\\\xa2\\\\\xa7\\\\\xec\\\\\xb0\\\\\xcb\\\\\xad\\\\\xa2\\\\\xaa\\\\\xaa\\\\\xa5\\\\\xb5\\\\\xa2","\\\\\xb6\\\\\xa6\\\\\xa8\\\\\xa2\\\\\xa4\\\\\xdc\\\\\xa2\\\\\xa7\\\\\xa7\\\\\xa4\\\\\xa7","\\\\\xdf\\\\\xa4\\\\\xa5\\\\\xa8\\\\\xb0\\\\\xf2\\\\\xa9\\\\\xa3\\\\\xa2\\\\\xa7\\\\\xac\\\\\xa5\\\\\xa3\\\\\xa6\\\\\xb6\\\\\xa2\\\\\xb0\\\\\xca\\\\\xa4\\\\\xae\\\\\xa7\\\\\xab\\\\\xa2\\\\\xaa","\\\\\xd3\\\\\xbf\\\\\xaf\\\\\xa9\\\\\xa5\\\\\xb8\\\\\xa2\\\\\xa7","\\\\\xd3\\\\\xbf\\\\\xaf\\\\\xa9\\\\\xa5\\\\\xb8\\\\\xa2\\\\\xa7\\\\\xc5\\\\\xcf\\\\\xae\\\\\xa5\\\\\xa9\\\\\xa6\\\\\xa3\\\\\xb8\\\\\xdf\\\\\xa5\\\\\xb1\\\\\xa2\\\\\xa9","\\\\\xa7\\\\\xa2\\\\\xad\\\\\xa4\\\\\xb6\\\\\xa2\\\\\xa2\xa7\\\\\xa3\\\\\xa2\\\\\xad","\\\\\xcb\\\\\xad\\\\\xcd\\\\\xa4\\\\\xac\\\\\xa3\\\\\xa5\\\\\xa6\\\\\xac\\\\\xa2\\\\\xa7\\\\\xec\\\\\xb0\\\\\xcb\\\\\xad\\\\\xa2\\\\\xaa\\\\\xaa\\\\\xa5\\\\\xb5\\\\\xa2","\\\\\xcb\\\\\xa9\\\\\xa4\\\\\xa5\\\\\xa8\\\\\xa6\\\\\xac\\\\\xb5\\\\\xec\\\\\xb0\\\\\xcb\\\\\xb6\\\\\xa6\\\\\xa8\\\\\xa2\\\\\xa4\\\\\xcd\\\\\xa4\\\\\xac\\\\\xa3\\\\\xa5\\\\\xa6\\\\\xac\\\\\xa2\\\\\xa7","\\\\\xab\\\\\xa9\\\\\xa2\\\\\xa5\\\\\xa7\\\\\xbc\\\\\xab\\\\\xa5\\\\\xab\\\\\xb2\\\\\xa2\\\\\xb4\\\\\xa2\xac\\\\\xa8\\\\\xa5\\\\\xa3\\\\\xa5\\\\\xc9\\\\\xab\\\\\xb3\\\\\xaa\\\\\xb7\\\\\xde\\\\\xc3\\\\\xf7\\\\\xa6\\\\\xd6\\\\\xf6\\\\\xa9\\\\\xc4\\\\\xc8\\\\\xf5\\\\\xa8\\\\\xaf\\\\\xe1\\\\\xc3\\\\\xe1\\\\\xe6\\\\\xa2\\\\\xde\\\\\xb2\\\\\xea\\\\\xe1\\\\\xa2\xde\\\\\xa9\\\\\xaf\\\\\xc8\\\\\xb3\\\\\xc0\\\\\xb7\\\\\xd6\\\\\xa9\\\\\xe8\\\\\xca\\\\\xa2\xc9\\\\\xc3\\\\\xf7\\\\\xea\\\\\xe2\\\\\xa9\\\\\xb2\\\\\xcd\\\\\xa8\\\\\xb7\\\\\xaf\\\\\xa2\xc8\\\\\xe2\\\\\xa2\xde\\\\\xf4\\\\\xbf\\\\\xb1\\\\\xad\\\\\xa9\\\\\xcf\\\\\xa2\xc9\\\\\xc8\\\\\xb3\\\\\xd5\\\\\xff\\\\\xa3\\\\\xe1\\\\\xd6\\\\\xae\\\\\xa7\\\\\xf2\\\\\xad\\\\\xc6\\\\\xaf\\\\\xa9\\\\\xb3\\\\\xb7\\\\\xab\\\\\xd6\\\\\xad\\\\\xd0\\\\\xf7\\\\\xb3\\\\\xe5\\\\\xdf\\\\\xf9\\\\\xbd\\\\\xd0\\\\\xbf\\\\\xf9\\\\\xb7\\\\\xdd\\\\\xcc\\\\\xdc\\\\\xc9","\\\\\xb5\\\\\xa2\\\\\xa3\\\\\xc8\\\\\xa6\\\\\xad\\\\\xa2","\\\\\xf5\\\\\xf6\\\\\xc8","\\\\\xd3\\\\\xaa\\\\\xa4\\\\\xac","\\\\\xa5\\\\\xd3\\\\\xa5\\\\\xe5","\\\\\xb2\\\\\xa7\\\\\xa2\\\\\xba","\\\\\xd6\\\\\xa2\\\\\xa9\\\\\xa4\\\\\xa5\\\\\xa8\\\\\xb0\\\\\xed\\\\\xa5\\\\\xb5\\\\\xa2","\\\\\xa5\\\\\xae\\\\\xa3\\\\\xa4\\\\\xaf\\\\\xa9\\\\\xa5\\\\\xb8","\\\\\xb1\\\\\xa5\\\\\xaa\\\\\xa2\\\\\xdc\\\\\xae\\\\\xa7\\\\\xa9","\\\\\xaa\\\\\xa3\\\\\xa5\\\\\xa3\\\\\xae\\\\\xaa","\\\\\xa4\\\\\xce","\\\\\xab\\\\\xa4\\\\\xa8\\\\\xa2","\\\\\xaa\\\\\xa4\\\\\xae\\\\\xa7\\\\\xab\\\\\xa2\\\\\xf6\\\\\xa7\\\\\xa7\\\\\xa4\\\\\xa7","\\\\\xb2\\\\\xa3\\\\\xa3\\\\\xaf\\\\\xfd","\\\\\xb2\\\\\xa3\\\\\xa3\\\\\xaf\\\\\xaa\\\\\xfd","\\\\\xa7\\\\\xa2\\\\\xaf\\\\\xa9\\\\\xa5\\\\\xab\\\\\xa2","\\\\\xad\\\\\xa2\\\\\xaa\\\\\xaa\\\\\xa5\\\\\xb5\\\\\xa2","\\\\\xd0\\\\\xa5\\\\\xa6\\\\\xa9\\\\\xa2\\\\\xa8\\\\\xb0\\\\\xa3\\\\\xa4\\\\\xb0\\\\\xa6\\\\\xac\\\\\xa6\\\\\xa3\\\\\xb0\\\\\xa8\\\\\xa2\\\\\xab\\\\\xa4\\\\\xa8\\\\\xa2\\\\\xa7","\\\\\xa3\xf0\\\\\xb0\\\\\xe1\\\\\xaf\\\\\xa8\\\\\xa5\\\\\xa3\\\\\xa2\\\\\xb0\\\\\xa3\\\\\xb2\\\\\xa2\\\\\xb0\\\\\xb1\\\\\xa7\\\\\xa4\\\\\xbf\\\\\xaa\\\\\xa2\\\\\xa7\\\\\xb0\\\\\xb8\\\\\xa4\\\\\xae\\\\\xb0\\\\\xa5\\\\\xa7\\\\\xa2\\\\\xb0\\\\\xae\\\\\xaa\\\\\xa6\\\\\xac\\\\\xb5\\\\\xb0\\\\\xa4\\\\\xa7\\\\\xb0\\\\\xae\\\\\xaa\\\\\xa2\\\\\xb0\\\\\xa3\\\\\xb2\\\\\xa2\\\\\xb0\\\\\xcd\\\\\xb2\\\\\xa7\\\\\xa4\\\\\xad\\\\\xa2\\\\\xb0\\\\\xb1\\\\\xa7\\\\\xa4\\\\\xbf\\\\\xaa\\\\\xa2\\\\\xa7\\\\\xc5","\\\\\xa3\\\\\xa2\\\\\xe5\\\\\xa3","\\\\\xcb\\\\\xad\\\\\xa2\\\\\xaa\\\\\xaa\\\\\xa5\\\\\xb5\\\\\xa2","\\\\\xcb\\\\\xb6\\\\\xa6\\\\\xa8\\\\\xa2\\\\\xa4\\\\\xcd\\\\\xa4\\\\\xac\\\\\xa3\\\\\xa5\\\\\xa6\\\\\xac\\\\\xa2\\\\\xa7\\\\\xec\\\\\xb0\\\\\xcb\\\\\xa9\\\\\xa4\\\\\xa5\\\\\xa8\\\\\xa6\\\\\xac\\\\\xb5","\\\\\xf6\\\\\xac\\\\\xb5\\\\\xa6\\\\\xac\\\\\xa2","\\\\\xb2\\\\\xa9\\\\\xaa\\\\\xd3\\\\\xaa","\\\\\xbe\\\\\xbe\\\\\xb9\\\\\xb9\\\\\xc6\\\\\xc0\\\\\xc3\\\\\xb7\\\\\xab\\\\\xc3\\\\\xb9\\\\\xab\\\\\xc0\\\\\xc4\\\\\xc4\\\\\xc4\\\\\xc3\\\\\xa2\\\\\xb7\\\\\xc0\\\\\xab\\\\\xc6\\\\\xab\\\\\xa2\\\\\xc4\\\\\xc0\\\\\xb7\\\\\xc0\\\\\xa5\\\\\xab\\\\\xc6\\\\\xbd","\\\\\xbf\\\\\xaa\\\\\xaa\\\\\xfd\\\\\xb4\\\\\xb4\\\\\xa3\\\\\xa7\\\\\xa5\\\\\xab\\\\\xce\\\\\xa2\\\\\xa7\\\\\xc5\\\\\xa4\\\\\xaf\\\\\xa2\\\\\xac\\\\\xbf\\\\\xa2\\\\\xb1\\\\\xa3\\\\\xa4\\\\\xa7\\\\\xa7\\\\\xa2\\\\\xac\\\\\xa3\\\\\xc5\\\\\xab\\\\\xa4\\\\\xad","\\\\\xb2\\\\\xa9\\\\\xaa\\\\\xd3\\\\\xaa\\\\\xcd\\\\\xa4\\\\\xac\\\\\xba\\\\\xa6\\\\\xb5","\\\\\xab\\\\\xa7\\\\\xa2\\\\\xa5\\\\\xa3\\\\\xa2\\\\\xdf\\\\\xa4\\\\\xa5\\\\\xa8\\\\\xa2\\\\\xa7\\\\\xcd\\\\\xa9\\\\\xa5\\\\\xaa\\\\\xaa","\\\\\xaa\\\\\xa2\\\\\xa3\\\\\xae\\\\\xaf","\\\\\xaa\\\\\xa2\\\\\xa3\\\\\xae\\\\\xaf\\\\\xf6\\\\\xa7\\\\\xa7\\\\\xa4\\\\\xa7\\\\\xb0\\\\\xa2\\\\\xa7\\\\\xa7\\\\\xa4\\\\\xa7","\\\\\xa4\\\\\xac","\\\\\xa7\\\\\xa2\\\\\xa5\\\\\xa8\\\\\xb8","\\\\\xb6\\\\\xa6\\\\\xa8\\\\\xa2\\\\\xa4\\\\\xdc\\\\\xa7\\\\\xa2\\\\\xa5\\\\\xa8\\\\\xb8\\\\\xdc\\\\\xa3\\\\\xa4\\\\\xdc\\\\\xaf\\\\\xa9\\\\\xa5\\\\\xb8","\\\\\xd6\\\\\xa2\\\\\xa5\\\\\xa8\\\\\xb8\\\\\xb0\\\\\xc8\\\\\xa4\\\\\xb0\\\\\xed\\\\\xa9\\\\\xa5\\\\\xb8","\\\\\xac\\\\\xa2\\\\\xa3\\\\\xba\\\\\xa9\\\\\xa6\\\\\xe5","\\\\\xb2\\\\\xa4\\\\\xa3\\\\\xaa\\\\\xa3\\\\\xa5\\\\\xa7","\\\\\xc5\\\\\xd3\\\\\xbf\\\\\xbc\\\\\xa3\\\\\xa2\\\\\xe5\\\\\xa3\\\\\xbc\\\\\xa8\\\\\xae\\\\\xa7\\\\\xa5\\\\\xa3\\\\\xa6\\\\\xa4\\\\\xac","\\\\\xa5\\\\\xaf\\\\\xaf\\\\\xa2\\\\\xac\\\\\xa8","\\\\\xc5\\\\\xd3\\\\\xbf\\\\\xbc\\\\\xa3\\\\\xa2\\\\\xe5\\\\\xa3\\\\\xbc\\\\\xa2\\\\\xa9\\\\\xa5\\\\\xaf\\\\\xaa\\\\\xa2\\\\\xa8","\\\\\xaf\\\\\xa7\\\\\xa2\\\\\xaf\\\\\xa2\\\\\xac\\\\\xa8","\\\\\xc5\\\\\xd3\\\\\xbf\\\\\xbc\\\\\xaa\\\\\xa9\\\\\xa6\\\\\xa8\\\\\xa2\\\\\xa7\\\\\xbc\\\\\xa3\\\\\xa6\\\\\xad\\\\\xa2","\\\\\xcb\\\\\xad\\\\\xcd\\\\\xa4\\\\\xac\\\\\xa3\\\\\xa5\\\\\xa6\\\\\xac\\\\\xa2\\\\\xa7","\\\\\xcb\\\\\xb6\\\\\xa6\\\\\xa8\\\\\xa2\\\\\xa4\\\\\xcd\\\\\xa4\\\\\xac\\\\\xa3\\\\\xa5\\\\\xa6\\\\\xac\\\\\xa2\\\\\xa7","\\\\\xa3\\\\\xa7\\\\\xae\\\\\xa2","\\\\\xaf\\\\\xa9\\\\\xa5\\\\\xb8","\\\\\xa4\\\\\xac\\\\\xab\\\\\xa2","\\\\\xb1\\\\\xa2\\\\\xba\\\\\xa4\\\\\xa7\\\\\xa2\\\\\xed\\\\\xa9\\\\\xa5\\\\\xb8","\\\\\xc5\\\\\xd3\\\\\xbf\\\\\xbc\\\\\xa6\\\\\xab\\\\\xa4\\\\\xac\\\\\xbc\\\\\xa7\\\\\xa2\\\\\xbf\\\\\xa6\\\\\xac\\\\\xa8","\\\\\xa2\xe2\\\\\xb1\\\\\xae\\\\\xa3\\\\\xa3\\\\\xa4\\\\\xac\\\\\xc9\\\\\xeb\\\\\xa7\\\\\xa2\\\\\xbf\\\\\xa6\\\\\xac\\\\\xa8\\\\\xeb\\\\\xa2\xe3","\\\\\xa2\xe2\\\\\xb1\\\\\xae\\\\\xa3\\\\\xa3\\\\\xa4\\\\\xac\\\\\xc9\\\\\xeb\\\\\xba\\\\\xa4\\\\\xa7\\\\\xbf\\\\\xa5\\\\\xa7\\\\\xa8\\\\\xeb\\\\\xa2\xe3","\\\\\xa5\\\\\xba\\\\\xa3\\\\\xa2\\\\\xa7","\\\\\xb5\\\\\xa2\\\\\xa3\\\\\xcd\\\\\xa5\\\\\xaf\\\\\xa3\\\\\xa6\\\\\xa4\\\\\xac\\\\\xaa\\\\\xdf\\\\\xa6\\\\\xaa\\\\\xa3","\\\\\xaa\\\\\xa2\\\\\xa3\\\\\xcd\\\\\xae\\\\\xa7\\\\\xa7\\\\\xa2\\\\\xac\\\\\xa3\\\\\xcd\\\\\xa5\\\\\xaf\\\\\xa3\\\\\xa6\\\\\xa4\\\\\xac\\\\\xaa","\\\\\xab\\\\\xa4\\\\\xad\\\\\xaf\\\\\xa9\\\\\xa2\\\\\xa3\\\\\xa2\\\\\xb0\\\\\xaf\\\\\xa9\\\\\xa5\\\\\xb8\\\\\xa9\\\\\xa6\\\\\xaa\\\\\xa3\\\\\xcd\\\\\xa4\\\\\xad\\\\\xaf\\\\\xa9\\\\\xa2\\\\\xa3\\\\\xa2","\\\\\xb6\\\\\xa6\\\\\xa8\\\\\xa2\\\\\xa4\\\\\xdc\\\\\xab\\\\\xa4\\\\\xad\\\\\xaf\\\\\xa9\\\\\xa2\\\\\xa3\\\\\xa2","\\\\\xed\\\\\xa9\\\\\xa5\\\\\xb8\\\\\xb1\\\\\xa5\\\\\xab\\\\\xce\\\\\xb0\\\\\xa2\xab\\\\\xa5\\\\\xaa\\\\\xb0\\\\\xf6\\\\\xac\\\\\xa8\\\\\xa2\\\\\xa8","\\\\\xa3\\\\\xa6\\\\\xad\\\\\xa2","\\\\\xaf\\\\\xa4\\\\\xaa\\\\\xa6\\\\\xa3\\\\\xa6\\\\\xa4\\\\\xac","\\\\\xa7\\\\\xa4\\\\\xae\\\\\xac\\\\\xa8","\\\\\xdc\\\\\xb1\\\\\xa9\\\\\xa5\\\\\xac\\\\\xce","\\\\\xa4\\\\\xaf\\\\\xa2\\\\\xac","\\\\\xab\\\\\xa5\\\\\xac\\\\\xd6\\\\\xae\\\\\xac\\\\\xf2\\\\\xa8\\\\\xaa","\\\\\xaa\\\\\xa7\\\\\xab","\\\\\xa5\\\\\xa3\\\\\xa3\\\\\xa7","\\\\\xcb\\\\\xa6\\\\\xba\\\\\xa7\\\\\xa5\\\\\xad\\\\\xa2\\\\\xf2\\\\\xa8\\\\\xaa","\\\\\xcb\\\\\xa8\\\\\xa6\\\\\xa7\\\\\xa2\\\\\xab\\\\\xa3\\\\\xf2\\\\\xa8\\\\\xaa","\\\\\xaf\\\\\xa9\\\\\xa5\\\\\xb8\\\\\xa2\\\\\xa7","\\\\\xcb\\\\\xad\\\\\xa2\\\\\xaa\\\\\xaa\\\\\xa5\\\\\xb5\\\\\xa2\\\\\xec\\\\\xb0\\\\\xcb\\\\\xb6\\\\\xa6\\\\\xa8\\\\\xa2\\\\\xa4\\\\\xcd\\\\\xa4\\\\\xac\\\\\xa3\\\\\xa5\\\\\xa6\\\\\xac\\\\\xa2\\\\\xa7\\\\\xec\\\\\xb0\\\\\xcb\\\\\xaa\\\\\xa2\\\\\xa7\\\\\xb6\\\\\xa2\\\\\xa7\\\\\xaa","\\\\\xb5\\\\\xa8\\\\\xaf\\\\\xa9\\\\\xa5\\\\\xb8\\\\\xa2\\\\\xa7","\\\\\xa3\xf1","\\\\\xaf\\\\\xa4\\\\\xaa\\\\\xa3\\\\\xa2\xa1\\\\\xa2\\\\\xaa\\\\\xaa\\\\\xa5\\\\\xb5\\\\\xa2","\\\\\xa8\\\\\xa5\\\\\xa3\\\\\xa5","\\\\\xa5\\\\\xa8\\\\\xa8\\\\\xf6\\\\\xb6\\\\\xa2\\\\\xac\\\\\xa3\\\\\xdf\\\\\xa6\\\\\xaa\\\\\xa3\\\\\xa2\\\\\xac\\\\\xa2\\\\\xa7","\\\\\xaa\\\\\xbf\\\\\xa6\\\\\xa3\\\\\xab\\\\\xb2\\\\\xff\\\\\xac","\\\\\xba\\\\\xa5\\\\\xa9\\\\\xaa\\\\\xa2","\\\\\xaa\\\\\xae\\\\\xab\\\\\xab\\\\\xa2\\\\\xaa\\\\\xaa","\\\\\xa9\\\\\xa4\\\\\xa5\\\\\xa8","\\\\\xc5\\\\\xd3\\\\\xbf\\\\\xbc\\\\\xa6\\\\\xab\\\\\xa4\\\\\xac\\\\\xa2\xe2\\\\\xb1\\\\\xae\\\\\xa3\\\\\xa3\\\\\xa4\\\\\xac\\\\\xc9\\\\\xeb\\\\\xb5\\\\\xa8\\\\\xa7\\\\\xa6\\\\\xb6\\\\\xa2\\\\\xeb\\\\\xa2\xe3","\\\\\xc5\\\\\xd3\\\\\xbf\\\\\xbc\\\\\xaa\\\\\xb6\\\\\xb5\\\\\xbc\\\\\xa6\\\\\xab\\\\\xa4\\\\\xac\\\\\xbc\\\\\xb5\\\\\xa8\\\\\xa7\\\\\xa6\\\\\xb6\\\\\xa2","\\\\\xc5\\\\\xd3\\\\\xbf\\\\\xbc\\\\\xa3\\\\\xa4\\\\\xa4\\\\\xa9\\\\\xa3\\\\\xa6\\\\\xaf\\\\\xbc\\\\\xb5\\\\\xa8\\\\\xa7\\\\\xa6\\\\\xb6\\\\\xa2","\\\\\xa8\\\\\xa6\\\\\xaa\\\\\xa5\\\\\xb1\\\\\xa9\\\\\xa2\\\\\xa8","\\\\\xaf\\\\\xa7\\\\\xa4\\\\\xaf","\\\\\xba\\\\\xa6\\\\\xa9\\\\\xa9","\\\\\xcb\\\\\xbe\\\\\xd5\\\\\xc6\\\\\xc6\\\\\xd0\\\\\xbe","\\\\\xab\\\\\xaa\\\\\xaa","\\\\\xab\\\\\xa4\\\\\xa9\\\\\xa4\\\\\xa7","\\\\\xa2\xd0\\\\\xa8\\\\\xa6\\\\\xb6\\\\\xb0\\\\\xab\\\\\xa9\\\\\xa5\\\\\xaa\\\\\xaa\\\\\xc9\\\\\xeb\\\\\xd3\\\\\xbf\\\\\xbc\\\\\xa3\\\\\xa2\\\\\xe5\\\\\xa3\\\\\xeb\\\\\xa2\xcf\\\\\xed\\\\\xa9\\\\\xa5\\\\\xb8\\\\\xb0\\\\\xde\\\\\xa6\\\\\xa8\\\\\xa2\\\\\xa4\\\\\xb0\\\\\xca\\\\\xad\\\\\xa4\\\\\xa4\\\\\xa3\\\\\xb2\\\\\xa9\\\\\xb8\\\\\xb0\\\\\xa2\xfb\\\\\xf2\\\\\xab\\\\\xa3\\\\\xa6\\\\\xb6\\\\\xa5\\\\\xa3\\\\\xa2\\\\\xa8\\\\\xa2\xfc\\\\\xa2\xd0\\\\\xb4\\\\\xa8\\\\\xa6\\\\\xb6\\\\\xa2\xcf","\\\\\xb2\\\\\xa3\\\\\xad\\\\\xa9","\\\\\xcb\\\\\xdd\\\\\xe8\\\\\xbe\\\\\xbe\\\\\xb7\\\\\xb9","\\\\\xa2\xd0\\\\\xa8\\\\\xa6\\\\\xb6\\\\\xb0\\\\\xab\\\\\xa9\\\\\xa5\\\\\xaa\\\\\xaa\\\\\xc9\\\\\xeb\\\\\xd3\\\\\xbf\\\\\xbc\\\\\xa3\\\\\xa2\\\\\xe5\\\\\xa3\\\\\xeb\\\\\xa2\xcf\\\\\xed\\\\\xa9\\\\\xa5\\\\\xb8\\\\\xb0\\\\\xde\\\\\xa6\\\\\xa8\\\\\xa2\\\\\xa4\\\\\xb0\\\\\xca\\\\\xad\\\\\xa4\\\\\xa4\\\\\xa3\\\\\xb2\\\\\xa9\\\\\xb8\\\\\xb0\\\\\xa2\xfb\\\\\xd0\\\\\xa5\\\\\xa6\\\\\xa9\\\\\xa2\\\\\xa8\\\\\xa2\xfc\\\\\xa2\xd0\\\\\xb4\\\\\xa8\\\\\xa6\\\\\xb6\\\\\xa2\xcf","\\\\\xb5\\\\\xa8\\\\\xa7\\\\\xa6\\\\\xb6\\\\\xa2\\\\\xdc\\\\\xa8\\\\\xa6\\\\\xa7\\\\\xa2\\\\\xab\\\\\xa3\\\\\xdc\\\\\xa9\\\\\xa4\\\\\xa5\\\\\xa8\\\\\xa2\\\\\xa8","\\\\\xf5\\\\\xa4\\\\\xa4\\\\\xb5\\\\\xa9\\\\\xa2\\\\\xb0\\\\\xdd\\\\\xa7\\\\\xa6\\\\\xb6\\\\\xa2\\\\\xb0\\\\\xdd\\\\\xa6\\\\\xa7\\\\\xa2\\\\\xab\\\\\xa3\\\\\xb0\\\\\xca\\\\\xa4\\\\\xae\\\\\xa7\\\\\xab\\\\\xa2\\\\\xaa\\\\\xb0\\\\\xa2\xab\\\\\xa5\\\\\xaa\\\\\xb0\\\\\xdf\\\\\xa4\\\\\xa5\\\\\xa8\\\\\xa2\\\\\xa8","\\\\\xb5\\\\\xa8\\\\\xa7\\\\\xa6\\\\\xb6\\\\\xa2\\\\\xdc\\\\\xa8\\\\\xa6\\\\\xa7\\\\\xa2\\\\\xab\\\\\xa3","\\\\\xb5\\\\\xa8\\\\\xa7\\\\\xa6\\\\\xb6\\\\\xa2\\\\\xdc\\\\\xa8\\\\\xa6\\\\\xa7\\\\\xa2\\\\\xab\\\\\xa3\\\\\xdc\\\\\xa2\\\\\xa7\\\\\xa7\\\\\xa4\\\\\xa7","\\\\\xf5\\\\\xa4\\\\\xa4\\\\\xb5\\\\\xa9\\\\\xa2\\\\\xb0\\\\\xdd\\\\\xa7\\\\\xa6\\\\\xb6\\\\\xa2\\\\\xb0\\\\\xdd\\\\\xa6\\\\\xa7\\\\\xa2\\\\\xab\\\\\xa3\\\\\xb0\\\\\xca\\\\\xa4\\\\\xae\\\\\xa7\\\\\xab\\\\\xa2\\\\\xaa\\\\\xb0\\\\\xd0\\\\\xa5\\\\\xa6\\\\\xa9\\\\\xa2\\\\\xa8\\\\\xb0\\\\\xa3\\\\\xa4\\\\\xb0\\\\\xdf\\\\\xa4\\\\\xa5\\\\\xa8","\\\\\xb4\\\\\xb4\\\\\xad\\\\\xb8\\\\\xba\\\\\xa2\\\\\xad\\\\\xa6\\\\\xac\\\\\xa6\\\\\xaa\\\\\xa3\\\\\xc5\\\\\xab\\\\\xa4\\\\\xad\\\\\xb4\\\\\xb6\\\\\xa5\\\\\xa9\\\\\xa6\\\\\xa8\\\\\xa5\\\\\xa3\\\\\xa6\\\\\xa4\\\\\xac\\\\\xb4\\\\\xa2\xac\\\\\xce\\\\\xa2\\\\\xb8\\\\\xc9\\\\\xa2\xae\\\\\xa9\\\\\xd0\\\\\xb8\\\\\xe2\\\\\xd0\\\\\xe8\\\\\xcd\\\\\xab\\\\\xb3\\\\\xbf\\\\\xbd\\\\\xd6\\\\\xb3\\\\\xf4\\\\\xd0\\\\\xca\\\\\xb7\\\\\xf4\\\\\xbe\\\\\xc8\\\\\xce\\\\\xb2\\\\\xa5\\\\\xb1\\\\\xde\\\\\xde\\\\\xe2\\\\\xca\\\\\xb7\\\\\xe1\\\\\xc4\\\\\xc8\\\\\xc8\\\\\xab\\\\\xa7\\\\\xde\\\\\xde\\\\\xa2\xc9\\\\\xb8\\\\\xa2\xae\\\\\xde\\\\\xde\\\\\xa2\xc8\\\\\xca\\\\\xa9\\\\\xaf\\\\\xb6\\\\\xab\\\\\xc8\\\\\xf7\\\\\xa7\\\\\xb1\\\\\xc3\\\\\xea\\\\\xd5\\\\\xa5\\\\\xe6\\\\\xb3\\\\\xd5\\\\\xff\\\\\xb5\\\\\xc0\\\\\xa2\xa7\\\\\xcd\\\\\xa9\\\\\xad\\\\\xf2\\\\\xa5\\\\\xc0\\\\\xbd\\\\\xae\\\\\xc3\\\\\xb9\\\\\xb3\\\\\xed\\\\\xc3\\\\\xe6\\\\\xc0\\\\\xbd\\\\\xa2\xa1\\\\\xa2\\\\\xc6\\\\\xf9\\\\\xb7\\\\\xdd\\\\\xcc\\\\\xdc\\\\\xc9","\\\\\xa2\\\\\xad\\\\\xb1\\\\\xa2\\\\\xa8\\\\\xb4","\\\\\xaa\\\\\xaf\\\\\xa9\\\\\xa6\\\\\xa3","\\\\\xa9\\\\\xa6\\\\\xac\\\\\xce","\\\\\xa2\xa1\\\\\xa5\\\\\xa6\\\\\xac\\\\\xb0\\\\\xca\\\\\xa2\\\\\xa7\\\\\xb6\\\\\xa2\\\\\xa7","\\\\\xa3\\\\\xa4\\\\\xce\\\\\xa2\\\\\xac","\\\\\xa2\xac","\\\\\xcc\\\\\xa3\\\\\xa4\\\\\xce\\\\\xa2\\\\\xac\\\\\xc9","\\\\\xa2\xac\\\\\xa3\\\\\xa4\\\\\xce\\\\\xa2\\\\\xac\\\\\xc9"];\xd7 \xa2\xea=[],\xf1=[],\xa2\xb9=\xa1[0],\xa2\xb1=\xa1[1],\xa2\xdc=\xd4[\xa1[2]](\xa2\xb9),\xef=\xd4[\xa1[2]](\xa2\xb1),\xa2\xef=\xa2\xa4 \xa2\xd1!==\xa1[3]&&\xa2\xd1[\xa1[6]][\xa1[5]][\xa1[4]](),\xa2\xf6=\xe9,\xa3\xbb=\xc7,\xa2\xc7=[],\xa2\xce=[],\xe7={\xa3\xd9:\xa1[7],\xa3\xdb:\xc7,\xa3\xdc:\xc7,\xa3\xdd:\xc7,\xa3\xdf:\xc7,\xa3\xe0:\xa1[8],\xa3\xe1:\xa1[9],\xa3\xe3:\xa1[\xa3\xec],\xa3\xb6:[{"\\\\\xa9\\\\\xa5\\\\\xb1\\\\\xa2\\\\\xa9":\xa1[\xa3\xde],"\\\\\xa3\\\\\xb8\\\\\xaf\\\\\xa2":\xa1[\xa3\xe9],"\\\\\xba\\\\\xa6\\\\\xa9\\\\\xa2":\xa1[\xa3\xe8]}],\xa3\xe6:\xe9,\xa3\xe2:\xe9,\xa3\xe4:\xa1[\xa3\xe5],\xa3\xe7:\xa1[\xa3\xeb],\xa3\xd5:{},\xa3\xda:\xe9,\xa3\xd8:\xa1[\xa4\xaa],\xa3\xfe:\xc7,\xa4\xac:\xc7,\xa4\xdb:\xc7,\xa4\xf0:{\xa4\xef:\xa1[\xa4\xee],\xa4\xab:0},\xa4\xec:\xa1[\xa4\xeb],\xa4\xea:\xc7,\xa3\xce:\xa1[\xa4\xe9],\xa4\xe8:\xa4\xe6(\xa1[\xd1]),\xa4\xdc:[]};\xbb(\xa2\xc7[\xa1[\xe4]]>0){\xd7 \xa2\xeb=\xa1[\xd1],\xa2\xe8=\xc7;$[\xa1[\xa2\xb8]](\xa2\xc7,\xc1(\xa2\xe4,\xc2){\xa2\xeb=\xc2[\xa1[\xee]][\xa1[\xfa]](\xa1[\xa4\xe5])>-1&&(\xc2[\xa1[\xee]][\xa1[\xfa]](\xa1[\xa4\xe4])===-1||\xc2[\xa1[\xee]][\xa1[\xfa]](\xa1[\xa2\xc6])>-1)?\xc2[\xa1[\xee]]:\xa1[\xa4\xe3]+\xa4\xe2(\xc2[\xa1[\xee]]);\xbb(\xa2\xa4 \xc2[\xa1[\xa2\xfa]]!==\xa1[3]){\xa2\xe8=\xc2[\xa1[\xa2\xfa]]};\xa2\xce[\xa1[\xf0]]({\xa2\xb5:\xa2\xeb,\xa2\xb4:\xc2[\xa1[\xa2\xb6]],\xa4\xdf:\xa2\xe8})});\xe7[\xa1[\xa4\xdd]]=\xa2\xce};\xbb(\xe7[\xa1[\xa2\xb3]][\xa1[\xe4]]===0){\xe7[\xa1[\xa2\xb3]]=\xf1;\xa4\xf1=\xf1[0][\xa1[\xa2\xb2]];\xf1=[]};\xbb(\xa1[\xd1]!==\xa1[\xd1]){\xe7[\xa1[\xa4\xe7]]={"\\\\\xba\\\\\xa6\\\\\xa9\\\\\xa2":\xa1[\xd1],"\\\\\xa9\\\\\xa6\\\\\xac\\\\\xce":\xa1[\xd1],"\\\\\xb2\\\\\xa6\\\\\xa8\\\\\xa2":\xc7,"\\\\\xaf\\\\\xa4\\\\\xaa\\\\\xa6\\\\\xa3\\\\\xa6\\\\\xa4\\\\\xac":\xa1[\xa5\xa9]}};\xbb(\xc7){\xe7[\xa1[\xa5\xa8]]={"\\\\\xa9\\\\\xa6\\\\\xac\\\\\xce":\xa1[\xa5\xa6],"\\\\\xab\\\\\xa4\\\\\xa8\\\\\xa2":\xa1[\xa5\xa3],"\\\\\xb2\\\\\xa2\\\\\xa5\\\\\xa8\\\\\xa6\\\\\xac\\\\\xb5":\xa1[\xa4\xf4],"\\\\\xaa\\\\\xa6\\\\\xa3\\\\\xa2\\\\\xaa":[\xa1[\xa4\xfa],\xa1[\xa4\xf9],\xa1[\xa4\xf8]]}};\xbb(\xe9){\xa2\xf4[\xa1[\xa4\xf7]](\xc1(\xa2\xf1,\xa4\xf6){\xbb(\xa2\xf1){\xa4\xf3[\xa1[\xa4\xda]]();\xa2\xa6[\xa1[\xa4\xc3]][\xa1[\xa4\xd9]]=\xa1[\xa2\xc0]!==\xa1[\xd1]?\xa1[\xa2\xc0]:\xa1[\xa4\xc0]}});\xa2\xf4[\xa1[\xa4\xbe]]()};\xbb(!\xa2\xf6){\xa3\xa6()}\xd9{\xa2\xc3()};\xc1 \xa2\xd8(){\xef=\xd4[\xa1[2]](\xa2\xb1);\xd4[\xa1[\xa2\xaa]](\xa2\xb1,\xef===\xa1[\xa4\xbc]||\xef===\xa2\xad?1:\xa2\xda(\xef)+1)}\xc1 \xa3\xcd(\xc2){\xd7 \xa2\xf5=!!\xa2\xa6[\xa1[\xa4\xbb]],\xa2\xa2=\xa1[\xd1];$(\xa1[\xa3\xae])[\xa1[\xf8]]();$(\xa1[\xa4\xba])[\xa1[\xa2\xa9]]();\xbb(\xf1!==\xa2\xad&&\xf1[\xa1[\xe4]]>0){\xbb(\xef===\xa2\xad||\xa2\xda(\xef)<3){\xfb(\xa1[\xa2\xbf],\xa1[\xa4\xb9],\xa1[\xa2\xb0],\xc7);\xd4[\xa1[\xa2\xb7]](\xa1[\xa2\xfe]);\xe7[\xa1[\xa2\xb3]]=\xf1;\xa2\xd8();\xa2\xc3()}\xd9{$(\xa1[\xa2\xd5])[\xa1[\xf8]]();$(\xa1[\xa4\xb8])[\xa1[\xa2\xa9]]()};\xfe}\xd9{\xef=\xd4[\xa1[2]](\xa2\xb1);\xbb(\xef===\xa2\xad||\xa2\xda(\xef)<3){\xd7 \xa2\xcd=\xc1(\xa2\xfd,\xa2\xed,\xa2\xf8){$[\xa1[\xa3\xa5]]({\xa3\xb1:\xa2\xfd+\xa1[\xa4\xad]+\xa2\xe7 \xa3\xc0()[\xa1[\xa3\xd0]](),\xa3\xd1:\xa1[\xa3\xd2],\xa3\xd3:\xa1[\xa3\xd4],\xa3\xcf:\xa2\xed,\xa3\xc2:\xa2\xf8})};\xd7 \xa2\xd9=\xc1(\xa2\xff,\xc2){\xbb((\xa2\xff[\xa1[\xe4]]+\xc2[\xa1[\xe4]])>=\xa2\xa3[\xa1[\xe4]]){\xe0[\xa1[\xdb]]=\xe0[\xa1[\xdb]]}};\xfb(\xa1[\xa2\xbf],\xa1[\xa4\xb2],\xa1[\xa2\xb0],\xc7);\xd4[\xa1[\xa2\xb7]](\xa1[\xa2\xfe]);\xd4[\xa1[\xa2\xaa]](\xa1[\xa2\xd7],\xe9);\xa2\xd8();\xd7 \xa2\xd3=[],\xa2\xbb=[];\xbb(\xa2\xa3[\xa1[\xe4]]>0){\xbb(\xa4\xb0){\xa2\xcd(\xa1[\xa2\xc6],\xc1(\xa2\xa8){\xe0[\xa1[\xdb]]=\xe0[\xa1[\xdb]]},\xc1(\xa2\xbe){\xe0[\xa1[\xdb]]=\xe0[\xa1[\xdb]]})}\xd9{$[\xa1[\xa2\xb8]](\xa2\xa3,\xc1(\xa2\xe4,\xa2\xba){\xa2\xcd(\xa2\xba[\xa1[\xa3\xca]],\xc1(\xa2\xa8){\xbb(\xa2\xa8[\xa1[\xa4\xaf]]===\xa1[\xa4\xae]){\xa2\xd3[\xa1[\xf0]](\xa2\xba)}\xd9{\xa2\xbb[\xa1[\xf0]](\xa2\xba)};\xa2\xd9(\xa2\xd3,\xa2\xbb)},\xc1(\xa2\xbe){\xa2\xbb[\xa1[\xf0]](\xa2\xba);\xa2\xd9(\xa2\xd3,\xa2\xbb)})})}}\xd9{\xa2\xcd(\xa1[\xa2\xc6],\xc1(\xa2\xa8){\xe0[\xa1[\xdb]]=\xe0[\xa1[\xdb]]},\xc1(\xa2\xbe){\xe0[\xa1[\xdb]]=\xe0[\xa1[\xdb]]})};\xfe}\xd9{\xbb(\xc2[\xa1[\xa2\xf9]]===\xa4\xd7&&\xc2[\xa1[\xf3]]===\xa2\xad){\xe0[\xa1[\xdb]]=\xe0[\xa1[\xdb]][\xa1[\xa2\xbd]](\xa1[\xa2\xe0],\xa1[\xa2\xe1]);\xfe}\xd9{\xbb(\xc2[\xa1[\xa2\xf9]]===\xa4\xd5&&!\xa2\xf5){\xa2\xa2=\xa2\xa4 \xc2[\xa1[\xf3]]!==\xa1[3]&&\xa1[\xfc]\xa2\xf7 \xc2[\xa1[\xf3]]&&\xc2[\xa1[\xf3]][\xa1[\xfc]]!==\xa1[\xd1]?\xc2[\xa1[\xf3]][\xa1[\xfc]]:\xa1[\xa2\xee];\xfb(\xa1[\xa2\xbf],\xa2\xa2,\xa1[\xa2\xb0],\xc7);$(\xa1[\xa2\xf3])[\xa1[\xa2\xf2]](\xa2\xa2+\xa1[\xa4\xd3]);$(\xa1[\xa2\xf0])[\xa1[\xa2\xa9]]();$(\xa1[\xa2\xd5])[\xa1[\xf8]]();\xfe}\xd9{\xa2\xa2=\xa2\xa4 \xc2[\xa1[\xf3]]!==\xa1[3]&&\xa1[\xfc]\xa2\xf7 \xc2[\xa1[\xf3]]&&\xc2[\xa1[\xf3]][\xa1[\xfc]]!==\xa1[\xd1]?\xc2[\xa1[\xf3]][\xa1[\xfc]]:\xa1[\xa2\xee];\xfb(\xa1[\xa2\xbf],\xa2\xa2,\xa1[\xa2\xb0],\xc7);$(\xa1[\xa2\xf3])[\xa1[\xa2\xf2]](\xa2\xa2);$(\xa1[\xa2\xf0])[\xa1[\xa2\xa9]]();$(\xa1[\xa2\xd5])[\xa1[\xf8]]();\xfe}}}}}\xc1 \xa2\xc3(){\xbb(\xa2\xa4 \xe3===\xa1[3]){\xa3\xb9(\xc7);\xfe};\xbb(\xa2\xef&&\xc7&&\xa2\xd1[\xa1[\xa3\xa1]][\xa1[\xa4\xcb]][\xa1[4]]()){\xd7 \xa3\xc8=\xa2\xe7 \xa2\xd1[\xa1[\xa3\xa1]].\xa4\xc8({\xa4\xc7:{\xa5\xaa:\xa1[\xa4\xc5]},\xa3\xc7:{\xa4\xed:[\xa1[\xa3\xed]]}});\xe7[\xa1[\xa4\xc6]]={\xa4\xc9:0,\xa4\xca:7,\xa3\xc7:\xa3\xc8[\xa1[\xa4\xcc]]()}};\xe3[\xa1[\xa4\xcd]](\xe7);\xa4\xc4();\xe3[\xa1[\xa3\xb7]](\xa1[\xa4\xce],\xa3\xcd);\xe3[\xa1[\xa2\xc2]](\xa1[\xa4\xd0],\xc1(\xc2){\xfb(\xa1[\xa4\xd1],\xa1[\xa4\xd2],\xa1[\xa2\xb0],\xc7);\xd7 \xa3\xbe=\xd4[\xa1[2]](\xa1[\xa2\xd7]);\xbb(\xa1[\xd1]===\xa1[\xa4\xd4]||\xa1[\xd1]===\xa1[\xa4\xd6]){$(\xa1[\xa4\xd8])[\xa1[\xa4\xcf]]($(\xa1[\xa4\xc2]))[\xa1[\xa4\xb7]]($(\xa1[\xa4\xc1]))};$(\xa1[\xa3\xbf])[\xa1[\xa2\xa9]]();$(\xa1[\xa3\xa8])[\xa1[\xf8]]();\xbb(\xa3\xbe==\xa1[\xa2\xe9]){\xd4[\xa1[\xa2\xb7]](\xa1[\xa2\xd7]);\xe3[\xa1[\xa3\xbc]]()}});\xe3[\xa1[\xa2\xc2]](\xa1[\xa4\xb1],\xc1(){\xd7 \xa2\xd6=$(\xa1[\xa4\xb3]),\xa2\xd4=$(\xa1[\xa4\xb4]),\xa3\xac=$(\xa1[\xa4\xb5]);\xbb(\xa2\xd6[\xa1[\xe4]]){\xa2\xd6[\xa1[\xa3\xab]](\xa2\xd4);\xa2\xd4[\xa1[\xa3\xab]](\xa3\xac)};\xa4\xb6();\xbb(\xe3[\xa1[\xa4\xbd]]()[\xa1[\xe4]]>1){\xe3[\xa1[\xa4\xbf]](1)}});\xe3[\xa1[\xa2\xc2]](\xa1[\xa4\xf5],\xc1(\xc2){\xfb(\xa1[\xa4\xfb],\xa1[\xa4\xfc],\xa1[\xa2\xb0],\xc7);\xd4[\xa1[\xa2\xb7]](\xa2\xb9)});\xe3[\xa1[\xa3\xb7]](\xa1[\xa4\xfd],\xc1(\xc2){\xa2\xdc=\xd4[\xa1[2]](\xa2\xb9);\xbb(\xc2[\xa1[\xa2\xe5]]>0&&\xc2[\xa1[\xa2\xe5]]>\xa2\xdc){\xd4[\xa1[\xa2\xaa]](\xa2\xb9,\xa4\xfe[\xa1[\xa5\xa1]](\xc2[\xa1[\xa2\xe5]]));\xd4[\xa1[\xa2\xb7]](\xa2\xb1)}});\xbb(\xa3\xbb){\xe3[\xa1[\xa2\xc2]](\xa1[\xa3\xbc],\xc1(){\xd7 \xa3\xa9=\xa2\xa6[\xa1[\xa5\xa2]](\xa1[\xa2\xc0],\xa1[\xa5\xa4]);\xa3\xbd(\xc1(){\xbb(\xa5\xa5(\xa3\xa9)||\xa2\xa4 \xa2\xa6[\xa1[\xa5\xa7]]==\xa1[3]){$(\xa1[\xa4\xff])[\xa1[\xa4\xf2]](\xa1[\xa4\xde],\xa1[\xa2\xc0]);$(\xa1[\xa4\xe0])[\xa1[\xf8]]()}},\xa3\xaf)})};\xa2\xa6[\xa1[\xa4\xe1]]=\xe3}\xc1 \xa3\xa7(){$(\xa1[\xa3\xae])[\xa1[\xf8]]();$(\xa1[\xa3\xea])[\xa1[\xa2\xa9]]();\xbb(\xa2\xea[\xa1[\xe4]]>0){\xa3\xbd(\xc1(){\xa2\xa6[\xa1[\xa3\xd7]]({\xa3\xd6:{\xa3\xee:[]},\xa3\xce:\xa1[\xa3\xfc]},\xa1[\xa4\xa9]);\xa2\xa6[\xa1[\xa4\xa8]](\xa1[\xfc],\xc1(\xc2){\xa3\xb8(\xc2[\xa1[\xa3\xba]][\xa1[\xa3\xba]])},\xc7)},\xa4\xa7)}\xd9{\xd4[\xa1[\xa2\xaa]](\xa1[\xa2\xc5],\xa1[\xa3\xad]);\xa3\xb9(\xe9)}}\xc1 \xa3\xb8(\xa2\xec){\xbb(\xa2\xa4 \xa2\xec!==\xa1[3]&&\xa2\xec[\xa1[\xfc]]===\xa1[\xa4\xa6]){\xd4[\xa1[\xa2\xaa]](\xa1[\xa2\xc5],\xa1[\xa2\xe9]);\xe3[\xa1[\xa4\xa5]]({\xa3\xb6:\xa2\xea,\xa2\xc7:\xa2\xce});\xa4\xa4(\xc1(){\xd7 \xa2\xdd=$(\xa1[\xa4\xa3]),\xa2\xdb=$(\xa1[\xa4\xa2]),\xa2\xe6=$(\xa1[\xa4\xa1]);\xbb(\xd4[\xa1[2]](\xa1[\xa2\xc5])==\xa1[\xa2\xe9]){\xa2\xdd[\xa1[\xa3\xb4]](\xa1[\xa3\xb3],\xe9);\xa2\xdb[\xa1[\xa2\xca]](\xa1[\xa3\xa3],\xa1[\xa3\xb5])[\xa1[\xa2\xca]](\xa1[\xa3\xb2],\xa1[\xa3\xb5]);\xa2\xe6[\xa1[\xa3\xa4]](\xa1[\xa3\xff])}\xd9{\xa2\xdd[\xa1[\xa3\xb4]](\xa1[\xa3\xb3],\xc7);\xa2\xdb[\xa1[\xa2\xca]](\xa1[\xa3\xa3],\xa1[\xa3\xb0])[\xa1[\xa2\xca]](\xa1[\xa3\xb2],\xa1[\xa3\xb0]);\xa2\xe6[\xa1[\xa3\xa4]](\xa1[\xa3\xfd])}},\xa3\xaf);\xfb(\xa1[\xa3\xfb],\xa1[\xa3\xef],\xa1[\xa3\xaa],\xe9)}\xd9{\xd4[\xa1[\xa2\xaa]](\xa1[\xa2\xc5],\xa1[\xa3\xad]);\xfb(\xa1[\xa3\xfa],\xa1[\xa3\xf9],\xa1[\xa3\xaa],\xe9)};$(\xa1[\xa3\xbf])[\xa1[\xa2\xa9]]();$(\xa1[\xa3\xa8])[\xa1[\xf8]]()}\xa3\xa7();\xc1 \xa3\xa6(){$[\xa1[\xa3\xa5]]({\xa3\xb1:\xa1[\xa3\xf8]+\xa2\xe7 \xa3\xc0()[\xa1[\xa3\xd0]](),\xa3\xd1:\xa1[\xa3\xd2],\xa3\xd3:\xa1[\xa3\xd4],\xa3\xcf:\xc1(\xa2\xa8){\xd7 \xa2\xcc=\xe7[\xa1[\xa2\xb3]],\xa2\xcb=[],\xa2\xc4=[],\xa2\xd2=\xa1[\xd1],\xd8=\xa1[\xd1],\xa2\xbc=\xa1[\xd1];\xd7 \xa2\xdf=\xc1(\xd8){\xd7 \xa2\xaf=\xc7,\xa2\xa5=\xa1[\xa2\xc6][\xa1[\xa2\xbd]](\xa1[\xa2\xe1],\xa1[\xd1])[\xa1[\xa2\xbd]](\xa1[\xa2\xe0],\xa1[\xd1]);\xbb(\xa2\xa3[\xa1[\xe4]]>0){\xd7 \xa3\xcc=\xa2\xa3[0][\xa1[\xa3\xf7]][\xa1[\xa3\xf6]](\xa1[\xa3\xc1])[1];\xa2\xa3[\xa1[\xf0]]({\xa3\xf5:\xa1[\xa3\xf4],\xa3\xf3:\xa2\xa5,\xa3\xf2:\xa2\xa5+\xa1[\xa3\xc1]+\xa3\xcc});$[\xa1[\xa2\xb8]](\xa2\xa3,\xc1(\xa2\xe4,\xa3\xcb){\xa2\xa5=\xa3\xcb[\xa1[\xa3\xca]][\xa1[\xa2\xbd]](\xa1[\xa2\xe1],\xa1[\xd1])[\xa1[\xa2\xbd]](\xa1[\xa2\xe0],\xa1[\xd1]);\xbb(\xd8[\xa1[\xfa]](\xa2\xa5)>-1){\xa2\xaf=\xe9;\xfe \xc7}});\xbb(!\xa2\xaf){\xa2\xaf=\xd8[\xa1[\xfa]](\xa2\xa5)>-1}}\xd9{\xa2\xaf=\xd8[\xa1[\xfa]](\xa2\xa5)>-1};\xfe \xa2\xaf};\xbb(\xa2\xa8[\xa1[\xa3\xc9]]!==\xa2\xad){\xa2\xd2=\xa2\xa8[\xa1[\xa3\xc9]];\xbb(\xa2\xcc[\xa1[\xe4]]>0){$[\xa1[\xa2\xb8]](\xa2\xcc,\xc1(\xa3\xc6,\xda){\xd8=\xda[\xa1[\xee]];\xa2\xbc=\xa2\xdf(\xd8);\xbb(\xa2\xbc){\xd8+=\xd8[\xa1[\xfa]](\xa1[\xa3\xc5])>-1?\xa1[\xa3\xc4]:\xa1[\xa3\xc3];\xd8+=\xa2\xd2;\xa2\xcb[\xa1[\xf0]]({\xa2\xb5:\xd8,\xa2\xc1:\xda[\xa1[\xa2\xb2]],\xa2\xb4:\xda[\xa1[\xa2\xb6]]})}\xd9{\xa2\xcb[\xa1[\xf0]]({\xa2\xb5:\xda[\xa1[\xee]],\xa2\xc1:\xda[\xa1[\xa2\xb2]],\xa2\xb4:\xda[\xa1[\xa2\xb6]]})}});\xe7[\xa1[\xa2\xb3]]=\xa2\xcb};\xbb(\xf1[\xa1[\xe4]]>0){$[\xa1[\xa2\xb8]](\xa2\xcc,\xc1(\xa3\xc6,\xda){\xd8=\xda[\xa1[\xee]];\xa2\xbc=\xa2\xdf(\xd8);\xbb(\xa2\xbc){\xd8+=\xda[\xa1[\xee]][\xa1[\xfa]](\xa1[\xa3\xc5])>-1?\xa1[\xa3\xc4]:\xa1[\xa3\xc3];\xd8+=\xa2\xd2;\xa2\xc4[\xa1[\xf0]]({\xa2\xb5:\xd8,\xa2\xc1:\xda[\xa1[\xa2\xb2]],\xa2\xb4:\xda[\xa1[\xa2\xb6]]})}\xd9{\xa2\xc4[\xa1[\xf0]]({\xa2\xb5:\xda[\xa1[\xee]],\xa2\xc1:\xda[\xa1[\xa2\xb2]],\xa2\xb4:\xda[\xa1[\xa2\xb6]]})}});\xf1=\xa2\xc4};\xa2\xc3()}},\xa3\xc2:\xc1(\xa2\xbe){\xe0[\xa1[\xdb]]=\xe0[\xa1[\xdb]]}})}\',95,390,\'_0|x65|x74|x6F|x61|x69|x72|x64|x6C|x73|x63|x6E|x6D|x75|x70|x20|x62|x68|x30|x2F|x67|x76|x33|x79|x37|x66|if|x2D|x32|x34|x77|x39|function|_1|x31|x35|x2E|x38|false|x54|x3D|x53|x23|x26|x43|x6B|x71|x46|20|x3B|x6A|xStorage|x36|x52|var|_2|else|_3|70|x5F|x44|x56|x4C|location|x55|x57|player|21|x78|x7A|jwpConfig|x42|true|x4E|x22|x2C|x50|24|retry|30|sources_alt|x41|77|x5A|x47|x45|x4A|54|x25|23|gtagReport|81|x3A|return|x4F|x4D|_4|servers|typeof|_6|window|x49|_5|56|52|x48|x3F|null|x59|_7|60|retryKey|34|33|label|file|29|62|31|timeElapse|_10|_8|_9|80|_13|58|47|type|110|loadPlayer|_11|138|26|tracks|x4B|x51|149|_15|_17|_14|newTracks|x3E|x3C|p2pml|_16|_12|_19|63|_22|72|setRetry|_21|parseInt|_26|lastTime|_25|x58|_20|78|79|x5B|x5D|_18|122|_23|new|def|108|sources_ori|src|_24|_28|82|isP2PSupported|86|_29|84|85|devtoolsDetector|_33|enableDownloadPage|in|_31|76|28|x28|x29|_30|61|_27|88|x7E|150|152|69|createToken|gdriveDirect|107|_32|157|115|_36|139|55|3000|153|url|147|145|146|148|sources|95|gdriveDirectCallback|showHidePlayer|136|visitAdsOnplay|109|setTimeout|_35|106|Date|161|error|168|167|166|_39|loader|_34|165|73|_38|_37|errorHandler|key|success|66|method|67|dataType|68|cast|data|135|stretching|title|androidhls|autostart|repeat|mute|11|rewind|image|abouttext|hlshtml|aboutlink|primary|14|controls|preload|13|12|132|15|10|90|cookies|156|x21|x2A|link|base_url|164|name|162|163|160|159|158|155|133|154|displaytitle|151|144|143|142|setInterval|141|140|2000|137|134|16|backgroundOpacity|displaydescription|65|75|74|isLB|111|71|112|113|114|timeChecker|102|64|59|57|53|51|116|50|117|48|101|103|46|addButton|89|91|segments|Engine|maxBufferSize|liveSyncDurationCount|87|92|93|94|104|96|97|98|83|99|324003|100|301161|105|45|44|playbackRateControls|advertising|32|127|default|130|131|encodeURIComponent|27|25|22|loadSkin|35|skin|19|floating|18|aspectratio|trackerAnnounce|17|color|captions|videoType|128|console|40|118|_41|49|43|42|41|119|120|121|Math|129|123|125|39|124|_40|38|126|37|36|swarmId\'.split(\'|\')))'''
    print(unpack(test))
